#!/bin/bash -l
# ------------------------------------------------------------------------------
# WRFOTRON v 2.2
# Christoph Knote (MBEES, University Augsburg)
# 09/2022 v 2.2
# 04/2022 v 2.1
# 09/2019 v 2.0
# 02/2016 v 1.0
# christoph.knote@med.uni-augsburg.de
# ------------------------------------------------------------------------------
#
# Main script to execute chained WRF simulations
# with meteo spinup and carried-on meteorology from
# previous runs.
#
# each run works like that:
#     |   spinup period    |     run period      |
#     | ------------------ | ------------------- |
# startDate             refDate               endDate
#
# * WPS and real.exe from startDate to endDate
#
# * WRF only simulation to spin-up meteorology from
#    startDate to refDate
#    --> creates restart file at refDate
#
# * WRF simulation from refDate to endDate
#    --> creates (output, obviously, but also) restart file at endDate
#
# * in case of met-only simulations, there's only one (nudged) run
#   from startDate to endDate

WRFOTRON_VERSION=22

# -----------------------------------------------------------------------------
# 1) read command line parameters
# -----------------------------------------------------------------------------

OPTIND=1

dryRun=false
while getopts "d" opt
do
  case $opt in
    d) dryRun=true ;;
    esac
done

shift $(($OPTIND - 1))

if [[ $# -lt 7 || $# -gt 8 ]]
then
  echo ""
  echo "Call with arguments <experiment> <year (YYYY)> <month (MM)> <day (DD)> <hour (hh)> <forecast time (h)> <spinup time (h)>"
  echo "                 or <experiment> <year (YYYY)> <month (MM)> <day (DD)> <hour (hh)> <forecast time (h)> <spinup time (h)> <PID of job dependency>"
  echo ""
  echo " * <experiment> can be a shortcut name (if experiment directory is found in subdirectory 'blueprints', "
  echo "                and you are calling it from the shell), or the absolute path to the experiment's settings directory."
  echo " * <spinup time> needs to be a multiple of meteoInc."
  echo ""
  echo " -d dry run, prepare only, do not submit or run"
  echo ""
  [[ "$0" != "$BASH_SOURCE" ]] && return 1 || exit 1
fi

experimentDir=$1
refYear=$2
refMonth=$3
refDay=$4
refHour=$5

fcstTime=$6
spinupTime=$7

isDependent=false
if [ $# -eq 8 ]
then
  isDependent=true
  depjobnr=$8
fi

experiment="missing"
if [ -d $experimentDir ]
then
  experiment=$(basename $experimentDir)
fi

if [ -d blueprints/$experimentDir ]
then
  experiment=$experimentDir
  experimentDir=$(pwd)/blueprints/$experiment
fi

# -----------------------------------------------------------------------------
# 2) sanity checks
# -----------------------------------------------------------------------------

if [ $experiment == "missing" ]
then
  echo "Experiment ($experimentDir) not found!"
  [[ "$0" != "$BASH_SOURCE" ]] && return 1 || exit 1
fi

# check for version of config
if [ ! -f ${experimentDir}/VERSION ]
then
  echo "No VERSION file found in ${experimentDir},"
  echo "experiment might have outdated config."
fi

if [ -f ${experimentDir}/VERSION ]
then
  config_version=$(cat ${experimentDir}/VERSION)
  if [ "$config_version" -lt "$WRFOTRON_VERSION" ]
  then
    echo "Config for experiment ${experiment} in ${experimentDir}"
    echo "seems outdated (VERSION file states ${config_version}, but WRFotron is ${WRFOTRON_VERSION}."
    [ -z "$PS1" ] && exit 1 || return 1
  fi
fi

# load configuration file
if [ ! -f ${experimentDir}/config.bash ]
then
  echo "Config file for experiment ${experiment} not found in ${experimentDir}!"
  [[ "$0" != "$BASH_SOURCE" ]] && return 1 || exit 1
fi

. ${experimentDir}/config.bash

# load machine-specific profile
if [ -f ${chainDir}/profile.bash ]
then
  . ${chainDir}/profile.bash
fi

# do it again, as variables might depend on profile.bash
. ${experimentDir}/config.bash

# check passed arguments
let dtTest="refHour % metInc"
if [ $dtTest -ne 0 ]
then
  echo "Hour needs to be multiple of $metInc."
  [[ "$0" != "$BASH_SOURCE" ]] && return 1 || exit 1
fi
let dtTest="fcstTime % metInc"
if [ $dtTest -ne 0 ]
then
  echo "Forecast time needs to be multiple of $metInc."
  [[ "$0" != "$BASH_SOURCE" ]] && return 1 || exit 1
fi
let dtTest="spinupTime % metInc"
if [ $dtTest -ne 0 ]
then
  echo "Spinup time needs to be multiple of $metInc."
  [[ "$0" != "$BASH_SOURCE" ]] && return 1 || exit 1
fi

# check external program availability
for prog in ncks ncatted nccopy
do
  hash $prog    >/dev/null 2>&1  || { echo "WRFotron needs '$prog'"; [[ "$0" != "$BASH_SOURCE" ]] && return 1 || exit 1; }
done

# check input directory availability
for dir in $chainDir $WPSDir $WRFDir $WRFChemDir $geogDir
do
  [ -d $dir ] || { echo "WRFotron could not find directory '$dir'"; [[ "$0" != "$BASH_SOURCE" ]] && return 1 || exit 1; }
done

# -----------------------------------------------------------------------------
# 3) calculation of date variables
# -----------------------------------------------------------------------------

dateTxt="$refYear-$refMonth-$refDay $refHour:00:00"

startYear=$(date -u --date="${dateTxt} ${spinupTime}hours ago" '+%Y')
startMonth=$(date -u --date="${dateTxt} ${spinupTime}hours ago" "+%m")
startDay=$(date -u --date="${dateTxt} ${spinupTime}hours ago" "+%d")
startHour=$(date -u --date="${dateTxt} ${spinupTime}hours ago" "+%H")

# do it again, to make sure to format single digit times correctly
refYear=$(date -u --date="${dateTxt}" '+%Y')
refMonth=$(date -u --date="${dateTxt}" "+%m")
refDay=$(date -u --date="${dateTxt}" "+%d")
refHour=$(date -u --date="${dateTxt}" "+%H")

endYear=$(date -u --date="${dateTxt} ${fcstTime}hours" "+%Y")
endMonth=$(date -u --date="${dateTxt} ${fcstTime}hours" "+%m")
endDay=$(date -u --date="${dateTxt} ${fcstTime}hours" "+%d")
endHour=$(date -u --date="${dateTxt} ${fcstTime}hours" "+%H")

# wrfout-file-style dates
startDate="${startYear}-${startMonth}-${startDay}_${startHour}:00:00"
refDate="${refYear}-${refMonth}-${refDay}_${refHour}:00:00"
endDate="${endYear}-${endMonth}-${endDay}_${endHour}:00:00"

# metInc in seconds
let metIncSec='metInc*3600'

# restartIntervals
let restartInc='fcstTime*60'
let restartIncSpinup='spinupTime*60'

# start day as Julian day
startJDay=$(date -u --date="${dateTxt} ${spinupTime}hours ago" '+%j')

# see if we actually need to spin up
test $spinupTime -gt 0
hasSpinup=$?

# -----------------------------------------------------------------------------
# 4) directory creation
# -----------------------------------------------------------------------------

restartDir=${restartRootDir}/${experiment}
stagingDir=${stagingRootDir}/${experiment}
archiveDir=${archiveRootDir}/${experiment}

runDir=${workDir}/${experiment}/${refDate}-${endDate}

prejobname=${experiment}${refYear}${refMonth}${refDay}.pre
spinupjobname=${experiment}${refYear}${refMonth}${refDay}.spinup
mainjobname=${experiment}${refYear}${refMonth}${refDay}.main
stagingjobname=${experiment}${refYear}${refMonth}${refDay}.staging
postjobname=${experiment}${refYear}${refMonth}${refDay}.post

startoutDir=$(pwd)

rm -rf ${runDir}

mkdir -p ${stagingDir} ${archiveDir} ${restartDir} ${workDir} ${runDir}

# -----------------------------------------------------------------------------
# 5) determine number of domains
# -----------------------------------------------------------------------------

# get the max_dom line from the namelist.wps file
max_dom=$(grep max_dom ${experimentDir}/namelist.wps)
# cut out the (only) number
max_dom=$(expr "$max_dom"  : '.*\([0-9]\).*')

# -----------------------------------------------------------------------------
# 6) create sed scripts to replace placeholders with actual variables
# -----------------------------------------------------------------------------

cd $runDir

# variables that are used in various scripts
cat > sedMiscScript << EOF
s/__geogDir__/${geogDir//\//\\/}/g
s/__workDir__/${workDir//\//\\/}/g
s/__runDir__/${runDir//\//\\/}/g
s/__configDir__//g
s/__preJobName__/${prejobname}/g
s/__spinupJobName__/${spinupjobname}/g
s/__mainJobName__/${mainjobname}/g
s/__stagingJobName__/${stagingjobname}/g
s/__postJobName__/${postjobname}/g
s/__refYear__/${refYear}/g
s/__refMonth__/${refMonth}/g
s/__refDay__/${refDay}/g
s/__refHour__/${refHour}/g
s/__refDate__/${refDate}/g
s/__refDateTxt__/${dateTxt}/g
s/__domains__/${max_dom}/g
s/__removeRunDir__/${removeRunDir}/g
s/__metIncSec__/${metIncSec}/g
s/__nNodesMain__/${nNodesMain}/g
s/__nTasksPerNodeMain__/${nTasksPerNodeMain}/g
s/__nNodesSpinup__/${nNodesSpinup}/g
s/__nTasksPerNodeSpinup__/${nTasksPerNodeSpinup}/g
EOF

# real run from -spinup to +fcstTime
cat > sedRealScript << EOF
s/__startYear__/${startYear}/g
s/__startMonth__/${startMonth}/g
s/__startDay__/${startDay}/g
s/__startHour__/${startHour}/g
s/__startDate__/${startDate}/g
s/__endYear__/${endYear}/g
s/__endMonth__/${endMonth}/g
s/__endDay__/${endDay}/g
s/__endHour__/${endHour}/g
s/__endDate__/${endDate}/g
s/__isRestart__/.false./g
s/__restartInterval__/${restartIncSpinup}/g
EOF

# spinup only runs from -spinup to 0
cat > sedSpinupScript << EOF
s/__startYear__/${startYear}/g
s/__startMonth__/${startMonth}/g
s/__startDay__/${startDay}/g
s/__startHour__/${startHour}/g
s/__startDate__/${startDate}/g
s/__endYear__/${refYear}/g
s/__endMonth__/${refMonth}/g
s/__endDay__/${refDay}/g
s/__endHour__/${refHour}/g
s/__endDate__/${refDate}/g
s/__isRestart__/.false./g
s/__restartInterval__/${restartIncSpinup}/g
s/__gridFdda__/1/g
EOF

# Actual runs start at 0 until +fcstTime
[ $hasSpinup -eq 0 ] && isRestart=".true." || isRestart=".false."
cat > sedRunScript << EOF
s/__startYear__/${refYear}/g
s/__startMonth__/${refMonth}/g
s/__startDay__/${refDay}/g
s/__startHour__/${refHour}/g
s/__startDate__/${refDate}/g
s/__endYear__/${endYear}/g
s/__endMonth__/${endMonth}/g
s/__endDay__/${endDay}/g
s/__endHour__/${endHour}/g
s/__endDate__/${endDate}/g
s/__isRestart__/${isRestart}/g
s/__restartInterval__/${restartInc}/g
s/__gridFdda__/0/g
EOF

# variables for WRF-Chem runs
cat > sedChemScript << EOF
s/__emissDir__/${emissDir//\//\\/}/g
s/__fireDir__/${fireDir//\//\\/}/g
s/__startJDay__/${startJDay}/g
s/__MEGANdir__/${MEGANdir//\//\\/}/g
EOF

# several runs of mozbc - IC (all (sub-)domains) and BC (only the first)
cat > sedMozOuterScript << EOF
s/__do_ic__/.true./g
s/__do_bc__/.true./g
s/__domain__/1/g
EOF

cat > sedMozInnerScript << EOF
s/__do_ic__/.true./g
s/__do_bc__/.false./g
EOF

if [ "$withChemistry" = "true" ]
then
  cat >> sedMiscScript < sedChemScript
fi

cat >> sedRealScript    < sedMiscScript
cat >> sedSpinupScript  < sedMiscScript
cat >> sedRunScript     < sedMiscScript

# -----------------------------------------------------------------------------
# 7) apply to run scripts
# -----------------------------------------------------------------------------

sed -f sedRealScript    < ${experimentDir}/namelist.wps         > namelist.wps.prep

sed -f sedRealScript    < ${experimentDir}/namelist.wrf         > namelist.wrf.real

sed -f sedSpinupScript  < ${experimentDir}/namelist.wrf         > namelist.wrf.spinup
sed -f sedRunScript     < ${experimentDir}/namelist.wrf         > namelist.wrf.run

[ "$withChemistry" = "true" ]               && sed -f sedRunScript < ${experimentDir}/namelist.chem >> namelist.wrf.real
[ "$withChemistry" = "true" ]               && sed -f sedRunScript < ${experimentDir}/namelist.chem >> namelist.wrf.run

# patch namelist for chemistry if needed
# meh - auxinput_interval_d might need max_domains values!
nmlChemPatchPath=${experimentDir}/namelist_timecontrol_chem_patch.wrf
[ -f ${nmlChemPatchPath} ]                  && sed -e "/! CHEM/ {" -e "r ${experimentDir}/namelist_timecontrol_chem_patch.wrf" -e "d" -e "}" -i namelist.wrf.real
[ -f ${nmlChemPatchPath} ]                  && sed -e "/! CHEM/ {" -e "r ${experimentDir}/namelist_timecontrol_chem_patch.wrf" -e "d" -e "}" -i namelist.wrf.run

# optional files to be processed
[ -f ${preScriptPath} ]                     && sed -f sedRealScript     < ${preScriptPath}                      > special_pre_actions.bash
[ -f ${postScriptPath} ]                    && sed -f sedRunScript      < ${postScriptPath}                     > special_post_actions.bash

# optional files for chemistry to be processed
[ -f ${experimentDir}/wesely.inp ]          && sed -f sedMiscScript     < ${experimentDir}/wesely.inp           > wesely.inp
[ -f ${experimentDir}/exo_coldens.inp ]     && sed -f sedMiscScript     < ${experimentDir}/exo_coldens.inp      > exo_coldens.inp
[ -f ${experimentDir}/megan_bio_emiss.inp ] && sed -f sedMiscScript     < ${experimentDir}/megan_bio_emiss.inp  > megan_bio_emiss.inp

[ -f ${experimentDir}/${emissInpFile} ]     && sed -f sedRealScript     < ${experimentDir}/${emissInpFile}      > anthro_emis.inp
[ -f ${experimentDir}/${fireInpFile} ]      && sed -f sedRealScript     < ${experimentDir}/${fireInpFile}       > fire_emis.inp

[ -f ${experimentDir}/${chembcInpFile} ]    && sed -f sedMozOuterScript < ${experimentDir}/${chembcInpFile}     > mozbc_outer.inp
[ -f ${experimentDir}/${chembcInpFile} ]    && sed -f sedMozInnerScript < ${experimentDir}/${chembcInpFile}     > mozbc_inner.inp

for job in pre main spinup staging post
do
  # add batch settings
  cat ${chainDir}/batch_preambles/${job} > ${job}.bash
  # add general preamble
  cat ${chainDir}/jobs/preamble.bash >> ${job}.bash
  # add jobs
  for jobscript in ${chainDir}/jobs/${job}/*bash
  do
  	# test if script has "chem" in name and withChemistry is true
  	# test if script has "ndown" in name and useNdown is true
  	jobname=$(basename $jobscript)
    if [ ! ${jobname#*chem} != ${jobname} -o "$withChemistry" = "true" ] && [ ! ${jobname#*ndown} != ${jobname} -o "$useNdown" = "true" ]
  	then
      cat >> ${job}.bash << EOF

echo -----------------------
echo $jobname
echo \$(date -u "+%Y-%m-%d %H:%M:%S")
echo -----------------------

EOF
      cat $jobscript        >> ${job}.bash
    fi
  done
  cat >> ${job}.bash << EOF

echo -----------------------
echo ... done
echo \$(date -u "+%Y-%m-%d %H:%M:%S")
echo -----------------------

EOF
  sed -f sedRealScript -i ${job}.bash
done

rm sedMiscScript sedChemScript sedRealScript sedSpinupScript sedRunScript sedMozOuterScript sedMozInnerScript

[ -f ${experimentDir}/iofields.met  ]           && cp ${experimentDir}/iofields.met  iofields.met
[ -f ${experimentDir}/iofields.chem ]           && cp ${experimentDir}/iofields.chem iofields.chem

for domain in $(seq -f "0%g" 1 ${max_dom})
do
  if [ -f ${experimentDir}/wrftrackinput_d${domain} ]
  then
    cp ${experimentDir}/wrftrackinput_d${domain} .
  fi
done

# -----------------------------------------------------------------------------
# 8) copy and augment config file
# -----------------------------------------------------------------------------

# copy machine specific profile
cp ${chainDir}/profile.bash .

cat > config.bash << EOF
#!/bin/bash -l

# --- Include machine profile ---

. profile.bash

EOF

# get config file contents
cat >> config.bash < ${experimentDir}/config.bash

cat >> config.bash << EOF

# --- Run-dependent variables ---

runDir=${runDir}
archiveDir=${archiveDir}
stagingDir=${stagingDir}
restartDir=${restartDir}

experiment=${experiment}

startDate=${startDate}
refDate=${refDate}
endDate=${endDate}

fcstTime=${fcstTime}
spinupTime=${spinupTime}

prejobname=${prejobname}
spinupjobname=${spinupjobname}
mainjobname=${mainjobname}
stagingjobname=${stagingjobname}
postjobname=${postjobname}

max_dom=${max_dom}

hasSpinup=${hasSpinup}

EOF

# -----------------------------------------------------------------------------
# 9) run submission
# -----------------------------------------------------------------------------

if [ "$dryRun" = "true" ]
then
  cd ${startoutDir}
  echo "${runDir}"
  [[ "$0" != "$BASH_SOURCE" ]] && return 1 || exit 1
fi

if [ "$isDependent" = "true" ]
then
  pretxt=$(${batchSubmitCommand} ${batchDepArgument/__id__/${depjobnr}} < pre.bash)
else
  pretxt=$(${batchSubmitCommand} ${batchHoldArgument} < pre.bash)
fi
prejobnr=$(echo $pretxt | sed -n "s/${batchPidSedCommand}/\1/p")

spinuptxt=$(${batchSubmitCommand} ${batchDepArgument/__id__/${prejobnr}} < spinup.bash)
spinupjobnr=$(echo $spinuptxt | sed -n "s/${batchPidSedCommand}/\1/p")

maintxt=$(${batchSubmitCommand} ${batchDepArgument/__id__/${spinupjobnr}} < main.bash)
mainjobnr=$(echo $maintxt | sed -n "s/${batchPidSedCommand}/\1/p")

stagingtxt=$(${batchSubmitCommand} ${batchDepArgument/__id__/${mainjobnr}} < staging.bash)
stagingjobnr=$(echo $stagingtxt | sed -n "s/${batchPidSedCommand}/\1/p")

posttxt=$(${batchSubmitCommand} ${batchDepArgument/__id__/${stagingjobnr}} < post.bash)
postjobnr=$(echo $posttxt | sed -n "s/${batchPidSedCommand}/\1/p")

if [ ! "$isDependent" = "true" ]
then
  ${batchReleaseCommand} ${prejobnr}
fi

echo $mainjobnr

# -----------------------------------------------------------------------------
# 10) cleanup
# -----------------------------------------------------------------------------

cd ${startoutDir}

