from wrf import cartopy_xlim, cartopy_ylim
import matplotlib.gridspec as gridspec

from matplotlib.offsetbox import AnchoredText

import datetime
import numpy as np

t2datetime = lambda x: datetime.datetime.strptime(x, '%Y-%m-%d_%H:%M:%S')

def wrf_time(nc):
    first = ''.join([x.decode('utf-8') for x in nc['Times'][0]])
    return t2datetime(first)

def total_prec(nc, prec_p):
    prec_vars = ['RAINC', 'RAINSH', 'RAINNC', 'SNOWNC', 'GRAUPELNC', 'HAILNC']
    start = t2datetime(nc.START_DATE)
    prec = np.sum([nc[var][0, :, :] for var in prec_vars], axis=0)
#
    if type(prec_p) == type(None):
        prec_p = prec
#
    if start == wrf_time(nc):
        return prec
    else:
        return prec-prec_p

# buffer 0.1 -> 10% border around the edges
def cartopy_lim_buffered(field, direction, buffer=0.1):
    if direction == "x":
        lims = cartopy_xlim(field)
    elif direction == "y":
        lims = cartopy_ylim(field)
    else:
        raise Exception("Direction not understood!")
    dlim = lims[1] - lims[0]
    lims[0] += 0.5 * buffer * dlim
    lims[1] -= 0.5 * buffer * dlim
    return lims

def make_fig_and_ax(plt, crs, add_horiz_cbar=False):
    fig      = plt.figure(figsize=(7,7))
#
    gs       = gridspec.GridSpec(2, 2, width_ratios=[20,1], height_ratios=[20,1], wspace=0.05, hspace=0.05)
#    
    ax       = fig.add_subplot(gs[0,0], projection=crs)
    ax_cbarv = fig.add_subplot(gs[0,1])
    ax_cbarh = None
    if add_horiz_cbar:
        ax_cbarh = fig.add_subplot(gs[1,0])
    return fig, ax, ax_cbarv, ax_cbarh

def add_title_and_description(nc, title, ax):
    '''
    title: including {init:s} for timestamp of run initialization, and {fcsthour:d} for forecast hour
    '''
    fcstHour = int((wrf_time(nc) - t2datetime(nc.START_DATE)).total_seconds()/3600.)
#    
    ax.set_title(title.format(init=t2datetime(nc.START_DATE).strftime("%d.%m.%Y %H:%M UTC"), 
                              fcsthour=fcstHour), fontdict={ 'fontsize':8, 'horizontalalignment': 'left'}, loc='left')
#
    texttime = AnchoredText(wrf_time(nc).strftime("%H UTC %a"), 
                            loc=2, pad=0.2, prop={'size': 10}, frameon=True)
    ax.add_artist(texttime)
#
    textcopyright = AnchoredText("$\copyright$ MBEES, Faculty of Medicine, University of Augsburg",
                        loc=4, pad=0.1, prop={'size': 6, 'color':'white'}, frameon=False)
    ax.add_artist(textcopyright)
